/*
 * Copyright (C) 2025 Samuel Weinig <sam@webkit.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <WebCore/RenderStyleConstants.h>
#include <WebCore/StyleValueTypes.h>

namespace WebCore {
namespace Style {

// <'border-image-repeat'> = [ stretch | repeat | round | space ]{1,2}
// https://drafts.csswg.org/css-backgrounds/#propdef-border-image-repeat
struct BorderImageRepeat {
    MinimallySerializingSpaceSeparatedSize<NinePieceImageRule> values { NinePieceImageRule::Stretch, NinePieceImageRule::Stretch };

    constexpr BorderImageRepeat(MinimallySerializingSpaceSeparatedSize<NinePieceImageRule> values)
        : values { values }
    {
    }
    constexpr BorderImageRepeat(NinePieceImageRule horizontalRule, NinePieceImageRule verticalRule)
        : values { horizontalRule, verticalRule }
    {
    }
    constexpr BorderImageRepeat(NinePieceImageRule rule)
        : values { rule, rule }
    {
    }
    constexpr BorderImageRepeat(CSS::Keyword::Stretch)
        : BorderImageRepeat { NinePieceImageRule::Stretch }
    {
    }
    constexpr BorderImageRepeat(CSS::Keyword::Repeat)
        : BorderImageRepeat { NinePieceImageRule::Repeat }
    {
    }
    constexpr BorderImageRepeat(CSS::Keyword::Round)
        : BorderImageRepeat { NinePieceImageRule::Round }
    {
    }
    constexpr BorderImageRepeat(CSS::Keyword::Space)
        : BorderImageRepeat { NinePieceImageRule::Space }
    {
    }

    constexpr NinePieceImageRule horizontalRule() const { return values.width(); }
    constexpr NinePieceImageRule verticalRule() const  { return values.height(); }

    constexpr bool operator==(const BorderImageRepeat&) const = default;
};
DEFINE_TYPE_WRAPPER_GET(BorderImageRepeat, values);

// MARK: - Conversion

template<> struct CSSValueConversion<BorderImageRepeat> { auto operator()(BuilderState&, const CSSValue&) -> BorderImageRepeat; };

} // namespace Style
} // namespace WebCore

DEFINE_TUPLE_LIKE_CONFORMANCE_FOR_TYPE_WRAPPER(WebCore::Style::BorderImageRepeat)
