; SOFT (Second-Order Functions and Theorems) Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SOFT")

(include-book "core")

(include-book "kestrel/event-macros/cw-event" :dir :system)
(include-book "std/system/irecursivep" :dir :system)
(include-book "std/system/well-founded-relation-plus" :dir :system)
(include-book "kestrel/utilities/er-soft-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ defsoft-implementation
  :parents (soft-implementation defsoft)
  :short "Implementation of @(tsee defsoft)."
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define ensure-wfrel-o< ((fn symbolp) ctx state)
  :returns (mv erp (nothing null) state)
  :short "Ensure that a function, if logic-mode and recursive,
          has @(tsee o<) as well-founded relation."
  :long
  (xdoc::topstring-p
   "For now, we only support logic-mode recursive second-order functions
    with the default well-founded relation (i.e. @(tsee o<)).
    This might be relaxed in the future.")
  (b* ((wrld (w state))
       ((unless (logicp fn wrld)) (value nil))
       ((unless (irecursivep fn wrld)) (value nil))
       (wfrel (well-founded-relation+ fn wrld))
       ((when (eq wfrel 'o<)) (value nil)))
    (er-soft+ ctx t nil
              "The well-founded relation of the recursive function ~x0 ~
               must be O<, but it is ~x1 instead."
              fn wfrel)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define ensure-defun-sk-rule-same-funvars ((fn symbolp) ctx state)
  :returns (mv erp (nothing "Always @('nil').") state)
  :mode :program
  :short "Ensure that a function, if introduced by @(tsee defun-sk),
          has an associated rewrite rule (the one generated by @(tsee defun-sk)
          that depends on exactly the same function variables
          that the function's matrix depends on."
  :long
  (xdoc::topstring-p
   "We collect
    the function variables that the @(tsee defun-sk) matrix depends on
    and the ones that the @(tsee defun-sk) rewrite rule depends on;
    we ensure that they are the same function variables.
    It seems unlikely that this check will ever fail in practice,
    but @(tsee defun-sk) allows custom rules (for universal quantifiers)
    that might somehow change the dependencies on function variables;
    for now we do not support this situation,
    but we might recosider this if some compelling example comes up.
    Unless the rewrite rule is a custom one,
    this check is always expected to pass.")
  (b* ((wrld (w state))
       ((unless (defun-sk-p fn wrld)) (value nil))
       (rule-name (defun-sk-rewrite-name fn wrld))
       (rule-body (formula rule-name nil wrld))
       (fn-matrix (defun-sk-matrix fn wrld))
       (rule-funvars (funvars-of-term rule-body wrld))
       (matrix-funvars (funvars-of-term fn-matrix wrld))
       ((when (set-equiv rule-funvars matrix-funvars)) (value nil))
       ((unless (eq (defun-sk-rewrite-kind fn wrld) :custom))
        (value (raise "Internal error: ~
                       the DEFUN-SK function ~x0 has a matrix ~x1
                       that depends on the function variables ~&2 ~
                       but a non-custom rewrite rule ~x3
                       that depends on the function variables ~&4.
                       This was not expected to happen."
                      fn fn-matrix matrix-funvars rule-body rule-funvars))))
    (er-soft+ ctx t nil
              "The DEFUN-SK function ~x0 has a matrix ~x1
               that depends on the function variables ~&2 ~
               but a custom rewrite rule ~x3
               that depends on the function variables ~&4."
              fn fn-matrix matrix-funvars rule-body rule-funvars)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define defsoft-fn (fn ctx state)
  :returns (mv erp event state)
  :mode :program
  :short "Generate the event submitted by @(tsee defsoft)."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @(tsee defsoft) macro records an ACL2 function
     as a (SOFT) second-order function.
     This macro will become the primary one
     to introduce second-order functions,
     and @(tsee defun2), @(tsee defchoose2), and @(tsee defun-sk2)
     will be redefined as @(tsee defun), @(tsee defchoose), and @(tsee defun-sk)
     followed by @(tsee defsoft).")
   (xdoc::p
    "The input @('fn') must be a symbol that denotes an existing function
     that is introduced by @(tsee defchoose),
     or otherwise is introduced by @(tsee defun-sk),
     or otherwise has an unnormalized body
     (which implies that is is introduced by @(tsee defun)).
     Note that @(tsee defun-sk) functions
     are internally introduced by @(tsee defun),
     so it is important to check for @(tsee defun-sk) first.
     Functions introduced by @(tsee defun) but without an unnormalized body
     (such as the built-in program-mode functions)
     are disallowed because we cannot calculate
     the function variables that such functions depend on.
     For the same reason,
     constrained functions introduced by @(tsee encapsulate) are disallowed.")
   (xdoc::p
    "We collect the function variables that the function depends on,
     directly or indirecty; there must be at least one.
     If the function is introduced by @(tsee defun-sk),
     we also ensure that the associated rewrite rule
     does not depend on additional function variables.
     If the function is recursive,
     we also ensure that the well-founded relation is @(tsee o<).")
   (xdoc::p
    "We print on screen an observation about the function being recorded
     and which function variables it depends on.
     This can be suppressed
     (e.g. when generating @(tsee defsoft) events programmatically)
     via @('(with-output :off observation ...)')."))
  (b* ((wrld (w state))
       ((unless (symbolp fn))
        (er-soft+ ctx t nil
                  "The input must be a symbol, but it is ~x0 instead."
                  fn))
       ((unless (function-symbolp fn wrld))
        (er-soft+ ctx t nil
                  "The symbol ~x0 must be a function symbol, ~
                   but it is not."
                  fn))
       ((unless (or (defchoosep fn wrld)
                    (defun-sk-p fn wrld)
                    (ubody fn wrld)))
        (er-soft+ ctx t nil
                  "The function ~x0 must ~
                   be introduced by DEFCHOOSE, ~
                   be introduced by DEFUN-SK, ~
                   or have a non-NIL unnormalized body."
                  fn))
       (funvars (cond ((defchoosep fn wrld) (funvars-of-choice-fn fn wrld))
                      ((defun-sk-p fn wrld) (funvars-of-quantifier-fn fn wrld))
                      (t (funvars-of-plain-fn fn wrld))))
       (funvars (remove-duplicates-eq funvars))
       ((unless (consp funvars))
        (er-soft+ ctx t nil
                  "The function ~x0 is not second-order:
                   it depends on no function variables, directly or indirectly."
                  fn))
       (table-event `(table second-order-functions ',fn ',funvars))
       ((er &) (ensure-wfrel-o< fn ctx state))
       ((er &) (ensure-defun-sk-rule-same-funvars fn ctx state))
       (state (acl2::io? observation
                         nil
                         state
                         (fn funvars)
                         (fms "SOFT: ~
                               recorded ~x0 as a second-order function ~
                               that depends on the function variables ~x1.~%"
                              (list (cons #\0 fn)
                                    (cons #\1 (acl2::sort-symbol-listp
                                               funvars)))
                              *standard-co*
                              state
                              nil))))
    (value
     `(progn
        ,table-event
        (value-triple ',fn)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection defsoft-macro-definition
  :short "Definition of the @(tsee defsoft) macro."
  :long
  "@(def defsoft)
   @(def acl2::defsoft)"

  (defmacro defsoft (fn)
    `(with-output
       :gag-mode nil
       :off ,(set-difference-eq acl2::*valid-output-names* '(error observation))
       :stack :push
       (make-event (defsoft-fn ',fn (cons 'defsoft ',fn) state)
                   :on-behalf-of :quiet!)))

  (defmacro acl2::defsoft (&rest args)
    `(defsoft ,@args)))
